package gov.va.med.mhv.usermgmt.service.adapter;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.GenericComposite;
import ca.uhn.hl7v2.model.Structure;
import ca.uhn.hl7v2.model.Type;
import ca.uhn.hl7v2.model.Varies;
import ca.uhn.hl7v2.model.v23.message.ACK;
import ca.uhn.hl7v2.model.v23.segment.RDF;
import ca.uhn.hl7v2.model.v23.segment.RDT;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;
import ca.uhn.hl7v2.parser.PipeParser;

import gov.va.med.mhv.usermgmt.transfer.Facility;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;

class LookupPatientDecoder {
	
	private static final String RDF = "RDF";

	private static final String RDT = "RDT";

	private final String firstName;

	private final String lastName;

	private final String birthDate;

	private final String ssn;

	public LookupPatientDecoder(String firstName, String lastName,
		Date birthDate, String ssn)
	{
		this.firstName = StringUtils.upperCase(firstName);
		this.lastName = StringUtils.upperCase(lastName);
		this.ssn = ssn;
		SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
		this.birthDate = df.format(birthDate);
	}

	public Patient decode(String msg) throws EncodingNotSupportedException,
		HL7Exception, DuplicatePatientException
	{
		PipeParser parser = new PipeParser();

		ACK ack_q02 = (ACK) parser.parse(msg);
		RDF rdf = (RDF) ack_q02.get(RDF);
		int columns = Integer.parseInt(rdf.getNumberOfColumnsPerRow()
			.getValue());

		Structure[] patients = null;
		try {
			patients = ack_q02.getAll(RDT);
		} catch (HL7Exception e) {
			// No patients
		}

		return extractPatient(patients, columns);
	}
	
	private Patient extractPatient(Structure[] patients, int columns) 
		throws HL7Exception, DuplicatePatientException 
	{
		if (patients == null) {
			return null;
		}
		List<Patient> exactMatchPatients = new ArrayList<Patient>();
		for (int i = 0; i < patients.length; i++) {
			if (isExactMatch(columns, (RDT) patients[i])) {
				exactMatchPatients.add(decodePatient(columns, 
					(RDT) patients[i]));
			} // else: Not an exact match
		}
		if (exactMatchPatients.size() > 1) {
			StringBuffer message = new StringBuffer(
				"Multiple ICN values found: ");
			int i = 0;
			for (Patient p : exactMatchPatients) {
				if (i > 0) {
					message.append(", ");
				}
				i++;
				message.append(p.getIcn());
			}
			throw new DuplicatePatientException(message.toString());
		}

		return (exactMatchPatients.size() == 1) ? exactMatchPatients.get(0) 
			: null;
	}

	private boolean isExactMatch(int columns, RDT encodedPatient) 
		throws HL7Exception
	{
		assert (encodedPatient != null);
		return (columns >= 5) // Must have enough columns to determine match
		     && equals(encodedPatient.getField(1), lastName)
			 && equals(encodedPatient.getField(3), firstName)
			 && equals(encodedPatient.getField(4), ssn)
			 && equals(encodedPatient.getField(5), 8, birthDate);
	}
	
	private Patient decodePatient(int columns, RDT encodedPatient) 
		throws HL7Exception
	{
		assert (encodedPatient != null);
		Patient patient = TransferObjectFactory.createPatient();
		
		if (columns >= 6) {
			patient.setIcn(fieldToString(encodedPatient.getField(6)));
		}
		
		if (columns >= 8) {
			List<String> stationNumbers = fieldToStrings(encodedPatient.
				getField(8));
			for (String stationNumber: stationNumbers) {
				if (!StringUtils.isBlank(stationNumber)) {
					Facility facility = TransferObjectFactory.createFacility();
					facility.setName(stationNumber);
					patient.addFacility(facility);
				}
			}
		} 
		
		return patient;
	}

	private boolean equals(Type[] field, String value) {
		return equals(value, fieldToString(field));
	}

	private boolean equals(Type[] field, int size, String value) {
		return equals(value, StringUtils.substring(fieldToString(field), 0, 8));
	}

	private boolean equals(String value1, String value2) {
		value1 = StringUtils.upperCase(value1);
		value2 = StringUtils.upperCase(value2);
		return StringUtils.equals(value1, value2)
			|| (StringUtils.isBlank(value1) && StringUtils.isBlank(value2));
	}

	private String fieldToString(Type[] field) {
		if (field == null) { 
			return null;
		}
		if (field.length == 0) {
			return null;
		}
		Type data = ((Varies) field[0]).getData();
		if (data == null) {
			return null;
		}
		return data.toString();
	}

	private List<String> fieldToStrings(Type[] field){
		List<String> strings = new ArrayList<String>();
		if (field == null) {
			return strings;
		}
		for (int k = 0; k < field.length; k++) {
			Type data = ((Varies) field[k]).getData();
			if (data instanceof GenericComposite) {
				GenericComposite gc = (GenericComposite) data;
				Type[] rpt = gc.getComponents();
				data = ((Varies) rpt[0]).getData();
			}
			if (data != null) {
				strings.add(data.toString());
			}
		}
		return strings;
	}


}
